#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_symbols.h"

#include "lf_mapper_map.h"

/*
 * local prototypes
 */
static void count_items(struct lf_mapfile *mp);
static void parse_xbar(struct lf_mapfile *mp, char *def);
static void parse_nic(struct lf_mapfile *mp, char *def);
static void verify_items(struct lf_mapfile *mp);

/*
 * Scan through the map file counting hosts and xbars
 */
static void
count_items(
  struct lf_mapfile *mp)
{
  lf_string_t buf;

  mp->nnic = 0;
  mp->nxbar = 0;

  mp->fp = fopen(mp->name, "r");
  if (mp->fp == NULL) {
    LF_ERROR(("Error opening %s", mp->name));
  }

  while (fgets(buf, sizeof(buf), mp->fp) != NULL) {
    if (strncmp(buf, "h ", 2) == 0) ++mp->nnic;
    else if (strncmp(buf, "s ", 2) == 0) ++mp->nxbar;
  }

  fclose(mp->fp);
  mp->fp = NULL;
  return;

 except:
  return;
}

/*
 * Parse text defining a crossbar
 */
static void
parse_xbar(
  struct lf_mapfile *mp,
  char *def)
{
  void *rp;
  char buf[80];
  int index;
  int nconn;
  struct lf_xbar *xp;
  int i;
  int xb2;
  int n2;
  int port2;
  int pn;
  char typ;
  char name[64];
  char xid[16];
  int rc;

  rc = sscanf(def, "s %s %s\n", xid, name);
  assert(rc == 2);

  /* register or lookup the xbar to get an index */
  index = lf_symtab_lookup_register(mp->xsyms, name);
  assert(index >= 0 && index < mp->nxbar);

  /* find this xbar struct and fill in stuff */
  xp = mp->xbar[index];
  LF_DUP_STRING(mp->xbar_names[index], name);

  /* assume 32 port xbars for now */
  xp->num_ports = 32;

  LF_CALLOC(xp->topo_ports, union lf_node *, xp->num_ports);
  LF_CALLOC(xp->topo_rports, int, xp->num_ports);

  /* next line is number of ports */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);

  rc = sscanf(buf, "%d", &nconn);
  assert(rc == 1);
  assert(nconn >= 0 && nconn <= xp->num_ports);

  /* now, nconn lines of connectivity */
  for (i=0; i<nconn; ++i) {
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
    rc = sscanf(buf, "%d %c %s %s %d", &pn, &typ, xid, name, &port2);
    assert(rc == 5);
    
    switch (typ) {
      case 's':
        xb2 = lf_symtab_lookup_register(mp->xsyms, name);
	assert(xb2 >= 0 && xb2 <= mp->nxbar);

	xp->topo_ports[pn] = LF_NODE(mp->xbar[xb2]);
	xp->topo_rports[pn] = port2;
	break;

      case 'h':
        n2 = lf_symtab_lookup_register(mp->nsyms, name);
	assert(n2 >= 0 && n2 <= mp->nnic);

	xp->topo_ports[pn] = LF_NODE(mp->nic[n2]);
	xp->topo_rports[pn] = port2;
	break;

      default:
	assert(0);
	break;
    }
  }

  /* finish out rest of xbar stanza */
  while (1) {
    char *wp[LF_STRING_LEN];
    int wc;

    /* get next line */
    rp = fgets(buf, sizeof(buf), mp->fp);

    /* done on blank line or EOF */
    if (rp == NULL || strlen(buf) <= 1) break;

    /* break it into words */
    line2words(buf, wp, "\r\n	 ", 0, &wc);

    if (wc < 1) break;

    /* Get XID */
    if (strcmp(wp[0], "ID") == 0) {
      assert(wc > 1);
      xp->xbar_id = atoi(wp[1]);


    } /* just skip any other lines in the NIC stanza */
  }

  if (xp->xbar_id == 0) {
    xp->num_ports = 16;
  }
  return;

 except:
  assert(0);
}

/*
 * parse a nic entry
 */
static void
parse_nic(
  struct lf_mapfile *mp,
  char *def)
{
  int rc;
  void *rp;
  char buf[80];
  int index;
  struct lf_nic *nicp;
  int port;
  int xbi;
  int xbp;
  int i;
  unsigned int mac[6];
  char name[63];
  lf_string_t sxid;
  int num_ports;

  rc = sscanf(def, "h - %s\n", name);
  assert(rc == 1);
  index = lf_symtab_lookup_register(mp->nsyms, name);
  assert(index >= 0 && index < mp->nnic);

  /* save the name from the map file */
  LF_DUP_STRING(mp->nic_names[index], name);

  nicp = mp->nic[index];

  /* number of ports */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);

  num_ports = atoi(buf);
  nicp->num_ports = num_ports;

  LF_CALLOC(nicp->topo_ports, union lf_node *, nicp->num_ports);
  LF_CALLOC(nicp->topo_rports, int, nicp->num_ports);

  for (i = 0; i < num_ports; i++) {

    /* a line defining our xbar connection */
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
  
    rc = sscanf(buf, "%d s %s %s %d\n", &port, sxid, name, &xbp);
    assert(rc == 4);
  
    /* Map file says how many ports defined, not "highest port index + 1" */
    if (port >= nicp->num_ports) {
      int new_num;
      int p;

      new_num = port + 1;
      nicp->topo_ports = (union lf_node **)
              realloc(nicp->topo_ports, new_num * sizeof(union lf_node *));
      if (nicp->topo_ports == NULL) LF_ERROR(("Error reallocing ports array"));
      nicp->topo_rports = (int *)
	realloc(nicp->topo_rports, new_num * sizeof(int));
      if (nicp->topo_rports == NULL) {
	LF_ERROR(("Error reallocing rports array"));
      }
      
      /* NULL out any new intervening pointers */
      for (p=nicp->num_ports; p < port; ++p) {
        nicp->topo_ports[p] = NULL;
      }
      
      nicp->num_ports = new_num;
    }
  
    xbi = lf_symtab_lookup_register(mp->xsyms, name);
    assert(xbi >= 0 && xbi < mp->nxbar);
  
    nicp->topo_ports[port] = LF_NODE(mp->xbar[xbi]);
    nicp->topo_rports[port] = xbp;
  }

  /*
   * Lazy way to do this - re-allocate a new NIC structure complete
   * with transceivers and replace this one.
   */
  {
    struct lf_nic *newnicp;

    newnicp = lf_alloc_generic_nic(nicp->num_ports);
    for (port=0; port<nicp->num_ports; ++port) {
      newnicp->topo_ports[port] = nicp->topo_ports[port];
      newnicp->topo_rports[port] = nicp->topo_rports[port];
    }

    lf_free_nic(nicp);
    nicp = newnicp;
    mp->nic[index] = nicp;

  }

  /* finish out rest of NIC stanza */
  while (1) {
    char *wp[LF_STRING_LEN];
    int wc;

    /* get next line */
    rp = fgets(buf, sizeof(buf), mp->fp);

    /* done on blank line or EOF */
    if (rp == NULL || strlen(buf) <= 1) return;

    /* break it into words */
    line2words(buf, wp, "\r\n	 ", 0, &wc);

    if (wc < 1) return;

    /* a line with MAC address ? */
    if (strcmp(wp[0], "address") == 0) {
      assert(wc > 1);
      rc = sscanf(wp[1], "%x:%x:%x:%x:%x:%x",
	    &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);

      if (rc != 6) {
	long long bigmac;
	rc = sscanf(wp[1], "%llx", &bigmac);
	assert(rc == 1);
	mac[0] = (bigmac >> 40) & 0xff;
	mac[1] = (bigmac >> 32) & 0xff;
	mac[2] = (bigmac >> 24) & 0xff;
	mac[3] = (bigmac >> 16) & 0xff;
	mac[4] = (bigmac >> 8) & 0xff;
	mac[5] = bigmac  & 0xff;
      }
      for (i=0; i<6; ++i) {
	nicp->mac_addr[i] = mac[i];
      }

    } /* just skip any other lines in the NIC stanza */
  }

  return;

 except:
  assert(0);
}


/*
 * Allocate structs for Hosts and Xbars, then parse the file filling them in.
 */
void
parse_items(struct lf_mapfile *mp)
{
  char buf[80];

  mp->fp = fopen(mp->name, "r");
  if (mp->fp == NULL) {
    perror(mp->name);
    exit(1);
  }

  while (fgets(buf, sizeof(buf), mp->fp) != NULL) {
    if (strncmp(buf, "h ", 2) == 0) parse_nic(mp, buf);
    else if (strncmp(buf, "s ", 2) == 0) parse_xbar(mp, buf);
  }

  fclose(mp->fp);
  mp->fp = NULL;
}

void
init_structs(struct lf_mapfile *mp)
{
  int i;

  LF_CALLOC(mp->nic, struct lf_nic *, mp->nnic);
  LF_CALLOC(mp->nic_names, char *, mp->nnic);
  for (i=0; i<mp->nnic; ++i) {
    LF_CALLOC(mp->nic[i], struct lf_nic, 1);
    mp->nic[i]->ln_type = LF_NODE_NIC;
  }

  LF_CALLOC(mp->xbar, struct lf_xbar *, mp->nxbar);
  LF_CALLOC(mp->xbar_names, char *, mp->nxbar);
  for (i=0; i<mp->nxbar; ++i) {
    LF_CALLOC(mp->xbar[i], struct lf_xbar, 1);
    mp->xbar[i]->ln_type = LF_NODE_XBAR;
  }

  return;

 except:
  assert(0);
}


/*
 * Scan through all connections, checking for symmetry
 */
static void
verify_items(
  struct lf_mapfile *mp)
{
  union lf_node *np;
  struct lf_xbar *xp;
  struct lf_xbar *xp2;
  struct lf_nic *nicp;
  int i;
  int p;
  int p2;

  /* check all xbar ports for symmetry */
  for (i=0; i<mp->nxbar; ++i) {
    xp = mp->xbar[i];

    for (p=0; p<xp->num_ports; ++p) {
      np = xp->topo_ports[p];
      p2 = xp->topo_rports[p];

      /* skip if no connection here */
      if (np == NULL) continue;

      switch (np->ln_type) {
	case LF_NODE_NIC:
	  nicp = LF_NIC(np);
	  if (nicp->topo_ports[p2] != LF_NODE(xp)) {
	    fprintf(stderr, "NIC pointer asymmetry x=%d, p=%d\n", i, p);
	  }
	  if (nicp->topo_rports[p2] != p) {
	    fprintf(stderr, "NIC port asymmetry x=%d, p=%d\n", i, p);
	  }
	  break;

	case LF_NODE_XBAR:
	  xp2 = LF_XBAR(np);

	  if (xp2->topo_ports[p2] != LF_NODE(xp)) {
	    fprintf(stderr, "port pointer asymmetry x=%d, p=%d\n", i, p);
	  }
	  if (xp2->topo_rports[p2] != p) {
	    fprintf(stderr, "port index asymmetry x=%d, p=%d\n", i, p);
	  }
	  break;

	default:
	  assert(0);
	  break;
      }
    }
  }

  /* check all host ports for symmetry */
  for (i=0; i<mp->nnic; ++i) {
    nicp = mp->nic[i];

    for (p = 0; p < nicp->num_ports; p++) {
      np = nicp->topo_ports[p];
      p2 = nicp->topo_rports[p];

      if (np == NULL) continue;		/* no-connects are possible */

      xp2 = LF_XBAR(np);
      if (xp2->topo_ports[p2] != LF_NODE(nicp)) {
        fprintf(stderr, "host xbar asymmetry x=%d, p=%d\n", i, p);
      }
      if (xp2->topo_rports[p2] != p) {
        fprintf(stderr, "host xbar port asymmetry x=%d, p=%d\n", i, p);
      }
    }
  }
}

/*
 * Load a gm_mapper map file
 */
struct lf_mapfile *
lf_load_mapper_map(
  char *name)
{
  struct lf_mapfile *mp;

  /* allocate mapfile struct */
  LF_CALLOC(mp, struct lf_mapfile, 1);
  LF_DUP_STRING(mp->name, name);

  mp->nsyms = lf_symtab_init();
  mp->xsyms = lf_symtab_init();

  count_items(mp);
  init_structs(mp);
  parse_items(mp);
  verify_items(mp);

  return mp;

 except:
  return NULL;
}
